/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package com.inet.taskplanner.logaction;

import java.io.IOException;
import java.util.List;

import com.inet.logging.LogLevel;
import com.inet.logging.LogManager;
import com.inet.logging.Logger;
import com.inet.taskplanner.server.api.action.ResultAction;
import com.inet.taskplanner.server.api.error.TaskExecutionException;
import com.inet.taskplanner.server.api.job.JobResultContainer;
import com.inet.taskplanner.server.api.result.Result;
import com.inet.taskplanner.server.api.result.ResultFlavor;
import com.inet.taskplanner.server.api.result.TextResult;

/**
 * This class implements the result action that is executed when there are jobs that create results that can be handled.<br>
 * It checks all results for text and prints the to the defined logger.
 */
public class LogResultAction extends ResultAction {

    private String   logName;
    private LogLevel logLevel;

    /**
     * Creates the action for the specified logger and log level
     * @param logName the name of the logger to be used
     * @param logLevel the level to log at
     */
    public LogResultAction( String logName, LogLevel logLevel ) {
        this.logName = logName;
        this.logLevel = logLevel;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void handle( List<JobResultContainer> results ) throws TaskExecutionException {
        // Create the logger
        Logger logger = LogManager.getLogger( logName );

        for( JobResultContainer resultContainer : results ) {
            // Get all results with the flavor "text" from the result containers
            List<Result> textResults = resultContainer.getResults( ResultFlavor.TEXT );
            for( Result result : textResults ) {
                TextResult textResult = (TextResult)result;
                try {
                    String text = textResult.getText();
                    if( text == null || text.isEmpty() ) {
                        continue;
                    }
                    switch( logLevel ) {
                        case DEBUG:
                            logger.debug( text );
                            break;
                        case ERROR:
                            logger.error( text );
                            break;
                        case INFO:
                            logger.info( text );
                            break;
                        case STATUS:
                            logger.status( text );
                            break;
                        case WARN:
                            logger.warn( text );
                            break;
                        default: // Do nothing
                    }
                } catch( IOException e ) {
                    throw new TaskExecutionException( e );
                }
            }
        }
    }
}
